#include <iostream>
#include <vector>
#include <tuple>
#include <stack>
#include <algorithm>
#include <cmath>
#include <numeric> 

using Point = std::pair<int, int>;

long long cross(const Point& O, const Point& A, const Point& B) {
    return 1LL * (A.first - O.first) * (B.second - O.second)
         - 1LL * (A.second - O.second) * (B.first - O.first);
}

bool cmp(const Point& A, const Point& B, const Point& start) {
    long long c = cross(start, A, B);
    if (c != 0) return c > 0;
    // Если на одной линии — ближе первее
    long long dx1 = A.first - start.first, dy1 = A.second - start.second;
    long long dx2 = B.first - start.first, dy2 = B.second - start.second;
    return dx1 * dx1 + dy1 * dy1 < dx2 * dx2 + dy2 * dy2;
}

std::vector<Point> convex_hull(std::vector<Point> pts) {
    if (pts.size() <= 1) return pts;

    // Найти стартовую точку (самая нижняя, затем самая левая)
    int start = 0;
    for (int i = 1; i < pts.size(); ++i) {
        if (pts[i].second < pts[start].second ||
           (pts[i].second == pts[start].second && pts[i].first < pts[start].first))
            start = i;
    }
    std::swap(pts[0], pts[start]);
    Point start_pt = pts[0];

    // Сортировка по полярному углу
    std::sort(pts.begin() + 1, pts.end(), [&](const Point& a, const Point& b) {
        return cmp(a, b, start_pt);
    });

    // Удалить дубликаты на одной линии (оставить самую дальнюю)
    int m = 1;
    for (int i = 1; i < pts.size(); ++i) {
        while (i < pts.size() - 1 && cross(start_pt, pts[i], pts[i+1]) == 0) ++i;
        pts[m++] = pts[i];
    }

    // Построение оболочки
    std::vector<Point> hull;
    for (int i = 0; i < m; ++i) {
        while (hull.size() >= 2 && cross(hull[hull.size()-2], hull.back(), pts[i]) <= 0)
            hull.pop_back();
        hull.push_back(pts[i]);
    }

    return hull;
}

std::pair<Point, Point> findFarthestPair(const std::vector<Point>& hull) {
    if (hull.size() < 2) {
        // По условию задачи, n >= 2, но на всякий случай
        return {hull[0], hull[0]};
    }

    long double maxDist = -1;
    Point bestA, bestB;

    for (size_t i = 0; i < hull.size(); ++i) {
        for (size_t j = i + 1; j < hull.size(); ++j) {
            long double dx = (long double)(hull[j].first - hull[i].first);
            long double dy = (long double)(hull[j].second - hull[i].second);
            long double dist = dx * dx + dy * dy; // можно без sqrt для сравнения

            if (dist > maxDist) {
                maxDist = dist;
                bestA = hull[i];
                bestB = hull[j];
            }
        }
    }
    return {bestA, bestB};
}

// Функция для нормализации коэффициентов (приведение к целым без общего делителя)
long long gcd(long long a, long long b) {
    a = std::abs(a);
    b = std::abs(b);
    if (b == 0) return a ? a : 1;
    return gcd(b, a % b);
}

void normalize(long long& a, long long& b, long long& c) {
    long long g = gcd(gcd(a, b), std::abs(c));
    if (g > 1) {
        a /= g;
        b /= g;
        c /= g;
    }
    // Приводим к каноническому виду: первый ненулевой из (a, b) > 0
    if (a < 0 || (a == 0 && b < 0)) {
        a = -a;
        b = -b;
        c = -c;
    }
}

int main() {
    int n;
    std::cin >> n;
    std::vector<Point> points(n);
    for (int i = 0; i < n; ++i) {
        std::cin >> points[i].first >> points[i].second;
    }

    auto hull = convex_hull(points);
    auto [P, Q] = findFarthestPair(hull);

    long long dx = (long long)Q.first - P.first;
    long long dy = (long long)Q.second - P.second;

    // Нормализуем направление нормали (dx, dy)
    long long g_norm = gcd(std::abs(dx), std::abs(dy));
    if (g_norm == 0) g_norm = 1; // на случай P == Q
    dx /= g_norm;
    dy /= g_norm;

    // Приведём к каноническому знаку
    if (dx < 0 || (dx == 0 && dy < 0)) {
        dx = -dx;
        dy = -dy;
    }

    // Теперь вычисляем c1 и c2 с уже нормализованными dx, dy
    long long c1 = -(dx * P.first + dy * P.second);
    long long c2 = -(dx * Q.first + dy * Q.second);

    // Выводим
    std::cout << dx << " " << dy << " " << c1 << "\n";
    std::cout << dx << " " << dy << " " << c2 << "\n";

    return 0;
}