#include <iostream>
#include <memory>
#include <random>
#include <chrono>
#include <format>
#include <mutex>
#include <thread>

const std::string COLLATZ_QUERY = "COLLAtz";
const std::string RANDOM_QUERY = "rAnDoM";

struct Number {
    std::string number;

    Number operator+(const int n) const {
        std::string ans;
        int carry = n;
        auto process = [&](char c) {
            int d = c - '0';
            d += carry;
            ans.push_back(d % 10 + '0');
            carry = d / 10;
        };
        for (char c : number)
            process(c);
        while (carry)
            process('0');
        return Number{.number = ans};
    }

    Number operator*(const int n) const {
        std::string ans;
        int carry = 0;
        auto process = [&](char c) {
            int d = c - '0';
            d *= n;
            d += carry;
            ans.push_back(d % 10 + '0');
            carry = d / 10;
        };
        for (char c : number)
            process(c);
        while (carry)
            process('0');
        return Number{.number = ans};
    }

    int operator%(const int n) const {
        int r = 0;
        for (auto it = number.rbegin(); it != number.rend(); ++it)
            r = (r * 10 + (*it - '0')) % n;
        return r;
    }

    Number operator/(const int n) const {
        std::string ans;
        int r = 0;
        for (auto it = number.rbegin(); it != number.rend(); ++it) {
            r = r * 10 + (*it - '0');
            if (r / n > 0 || !ans.empty())
                ans.push_back('0' + r / n);
            r %= n;
        }
        std::reverse(ans.begin(), ans.end());
        return Number{.number = ans};
    }

    bool operator<(const Number &other) const {
        if (number.size() != other.number.size())
            return number.size() < other.number.size();
        for (ssize_t i = ssize(number) - 1; i >= 0; --i)
            if (number[i] != other.number[i])
                return number[i] < other.number[i];
        return false;
    }

    template<class Int>
    static Number fromInt(const Int& n) {
        Number ans{
            .number = std::to_string(n),
        };
        std::reverse(ans.number.begin(), ans.number.end());
        return ans;
    }
};

std::ostream& operator<<(std::ostream& out, const Number& n) {
    if (n.number.empty())
        return out << '0';
    for (ssize_t i = ssize(n.number) - 1; i >= 0; --i)
        out << n.number[i];
    return out;
}

struct Strategy {
    virtual void restart() = 0;
    virtual bool choose_collatz(const Number& n) = 0;
    virtual std::string get_name() const = 0;
};

bool interact_test(std::shared_ptr<Strategy> strategy) {
    int64_t n;
    std::cin >> n;
    Number x = Number::fromInt(n);
    static const Number ONE = Number::fromInt(1);
    while (ONE < x) {
        bool choice = strategy->choose_collatz(x);
        std::cout << (choice ? COLLATZ_QUERY : RANDOM_QUERY) << std::endl;
        std::cin >> x.number;
        if (x.number == "0")
            return false;
        std::reverse(x.number.begin(), x.number.end());
    }
    return true;
}

void interact_multitest(std::shared_ptr<Strategy> strategy) {
    size_t t;
    std::cin >> t;
    for (size_t test_index = 1; test_index <= t; ++test_index) {
        strategy->restart();
        if (!interact_test(strategy))
            return;
    }
}

struct UpControlStrategy : Strategy {
    size_t window;
    double part;
    UpControlStrategy(size_t window, double part) : window(window), part(part) {
    }
    void restart() override {}
    bool choose_collatz(const Number& n) override {
        static const Number ONE = Number::fromInt(1);
        size_t ups = 0;
        Number x = n;
        for (size_t i = 0; i < window; ++i) {
            if (x % 2 == 0) {
                x = x / 2;
            } else if (ONE < n) {
                x = x * 3 + 1;
                ++ups;
            }
            if (ups <= part * (i + 1))
                return true;
        }
        return false;
    }
    std::string get_name() const override {
        return std::format("up_control({}, {:.3f})", window, part);
    }
};

int main(int argc, char* argv[]) {
    std::shared_ptr<Strategy> strategy = std::make_shared<UpControlStrategy>(17, 0.36);
    interact_multitest(strategy);
    return 0;
}
