#include <iostream>
#include <vector>
#include <algorithm>
#include <numeric>
#include <cmath>
#include <unordered_map>
#include <map>
#include <queue>

#define all(x) x.begin(), x.end()

using namespace std;

const double eps = 1e-7;
const double PI = 3.14159265359;

using ll = long long;

template <typename T>
bool equal(const T& a, const T& b) {
    return abs(a - b) < eps;
}

template <typename T>
struct Point {
    T x, y;
    bool NIL;
    Point() :x(T()), y(T()), NIL(true) {}
    Point(T x, T y) :x(x), y(y), NIL(false) {}
    Point<T> operator -(const Point<T>& other) const {
        return Point<T>(x - other.x, y - other.y);
    }
    Point<T> operator +(const Point<T>& other) const {
        return Point<T>(x + other.x, y + other.y);
    }
    Point<T>& operator +=(const Point<T>& other) {
        x += other.x;
        y += other.y;
        return *this;
    }
    Point<T> operator *(T k) const {
        return Point(x * k, y * k);
    }
    Point<T>& operator *=(T k) {
        x *= k;
        y *= k;
        return *this;
    }
    bool operator ==(const Point<T>& other) const {
        return x == other.x && y == other.y;
    }
    bool operator !=(const Point<T>& other) const {
        return !(*this == other);
    }
    bool operator <(const Point<T>& other) const {
        return make_pair(x, y) < make_pair(other.x, other.y);
    }
    double len() const {
        return hypot(x, y);
    }
    T len2() const {
        return x * x + y * y;
    }
    Point<T> norm() const {
        return Point<T>(x / len(), y / len());
    }
    Point<T> tolen(double l) const {
        return norm() * l;
    }
    bool isNIL() const {
        return NIL;
    }
};

template <typename T>
double len(const Point<T>& a) {
    return hypot(a.x, a.y);
}

template <typename T>
T dP(const Point<T>& a, const Point<T>& b) {
    return a.x * b.x + a.y * b.y;
}

template <typename T>
T cP(const Point<T>& a, const Point<T>& b) {
    return a.x * b.y - a.y * b.x;
}

template <typename T>
double angle(const Point<T>& a, const Point<T>& b) {
    return atan2(cP(a, b), dP(a, b));
}

template <typename T>
double polangle(const Point<T>& a) {
    double ans = angle(Point<T>(1, 0), a);
    if (ans < 0) {
        ans += 2 * PI;
    }
    return ans;
}

template <typename T>
bool pointOnLine(const Point<T>& p, const Point<T>& a, const Point<T>& b) {
    return cP(p - a, b - a) == 0;
}

template <typename T>
bool pointOnRay(const Point<T>& p, const Point<T>& a, const Point<T>& b) {
    return cP(p - a, b - a) == 0 && dP(p - a, b - a) >= 0;
}

template <typename T>
bool pointOnSegment(const Point<T>& p, const Point<T>& a, const Point<T>& b) {
    return pointOnRay(p, a, b) && pointOnRay(p, b, a);
}

template <typename T>
double distToLine(const Point<T>& p, const Point<T>& a, const Point<T>& b) {
    return abs(cP(p - a, b - a) * 1.0 / len(a - b));
}

template <typename T>
double distToRay(const Point<T>& p, const Point<T>& a, const Point<T>& b) {
    if (dP(p - a, b - a) < 0) {
        return len(p - a);
    } else {
        return distToLine(p, a, b);
    }
}

template <typename T>
double distToSegment(const Point<T>& p, const Point<T>& a, const Point<T>& b) {
    if (dP(p - a, b - a) < 0) {
        return len(p - a);
    } else if (dP(p - b, a - b) < 0) {
        return len(p - b);
    } else {
        return distToLine(p, a, b);
    }
}

template <typename T>
bool doProjectionsCross(T a, T b, T c, T d) {
    if (a > b) {
        swap(a, b);
    }
    if (c > d) {
        swap(c, d);
    }
    return max(a, c) <= min(b, d);
}

template <typename T>
int sign(T x) {
    if (x > T(0)) {
        return 1;
    } else if (x == T(0)) {
        return 0;
    } else {
        return -1;
    }
}

template <typename T>
bool isSameSign(T a, T b) {
    int sa = sign(a), sb = sign(b);
    return !(sa * sb == -1);
}

template <typename T>
bool isDiffSign(T a, T b) {
    int sa = sign(a), sb = sign(b);
    return !(sa * sb == 1);
}

template <typename T>
bool doSegmentsCross(const Point<T>& a, const Point<T>& b, const Point<T>& c, const Point<T>& d) {
    return isDiffSign(cP(b - a, c - a), cP(b - a, d - a)) &&
        isDiffSign(cP(d - c, a - c), cP(d - c, b - c)) &&
        doProjectionsCross(a.x, b.x, c.x, d.x) && doProjectionsCross(a.y, b.y, c.y, d.y);
}

template <typename T>
bool pointInAngle(const Point<T>& p, const Point<T>& a, const Point<T>& o, const Point<T>& b) {
    return isSameSign(cP(p - o, a - o), cP(b - o, a - o)) &&
        isSameSign(cP(p - o, b - o), cP(a - o, b - o));
}

template <typename T>
istream& operator >>(istream& in, Point<T>& p) {
    p.NIL = false;
    in >> p.x >> p.y;
    return in;
}

template <typename T>
ostream& operator <<(ostream& out, const Point<T>& p) {
    out << p.x << ' ' << p.y;
    return out;
}

template <typename T>
struct Line {
    T A, B, C;
    Line() :A(T()), B(T()), C(T()) {}
    Line(T A, T B, T C) :A(A), B(B), C(C) {}
    Line(const Point<T>& a, const Point<T>& b) {
        /*
        (x - x1) / (x2 - x1) = (y - y1) / (y2 - y1)
        xy2 - xy1 - x1y2 = yx2 - yx1 - y1x2
        x(y2 - y1) + y(x1 - x2) + y1x2 - x1y2 = 0
        */
        A = b.y - a.y;
        B = a.x - b.x;
        C = -cP(a, b);
    }
    Point<T> norm() const {
        return Point<T>(A, B);
    }
    Point<T> coll() const {
        return Point<T>(-B, A);
    }
    bool pointOnLine(const Point<T>& p) const {
        return A * p.x + B * p.y + C == 0;
    }
    double distToLine(const Point<T>& p) const {
        return abs(A * p.x + B * p.y + C) / hypot(A, B);
    }
};

template <typename T>
bool areLinesEqual(const Line<T>& a, const Line<T>& b) {
    return a.A * b.B == a.B * b.A && a.A * b.C == a.C * b.A;
}

template <typename T>
istream& operator >>(istream& in, Line<T>& l) {
    in >> l.A >> l.B >> l.C;
    return in;
}

template <typename T>
ostream& operator <<(ostream& out, const Line<T>& l) {
    out << l.A << ' ' << l.B << ' ' << l.C;
    return out;
}

template <typename T>
Point<T> crossLine(const Line<T>& l1, const Line<T>& l2) {
    if (l1.A * l2.B - l2.A * l1.B == 0) {
        return Point<T>();
    }
    T x = -(l1.C * l2.B - l2.C * l1.B) / (l1.A * l2.B - l2.A * l1.B);
    T y = -(l1.A * l2.C - l2.A * l1.C) / (l1.A * l2.B - l2.A * l1.B);
    return Point<T>(x, y);
}

template <typename T>
struct Circle {
    Point<T> c;
    double r;
    Circle() :c(Point<T>()), r(0) {}
    Circle(const Point<T>& c, double r) :c(c), r(r) {}
};

template <typename T>
istream& operator >>(istream& in, Circle<T>& o) {
    in >> o.c >> o.r;
    return in;
}

template <typename T>
vector<Point<T>> crossLineCircle(const Line<T>& l, const Circle<T>& o) {
    double h = l.distToLine(o.c);
    if (h >= o.r + eps) {
        return {};
    }
    Point<T> norm = l.norm().tolen(h);
    if (!equal(l.distToLine(o.c + norm), T(0))) {
        norm *= -1;
    }
    Point<T> base = o.c + norm;
    if (equal(h, o.r)) {
        return { base };
    }
    double d = sqrt(o.r * o.r - h * h);
    Point<double> coll = l.coll().tolen(d);
    return { base + coll, base + coll * (-1) };
}

template <typename T>
vector<Point<T>> tangentPoints(const Point<T>& p, const Circle<T>& o) {
    if (o.pointOnCircle(p)) {
        return { p };
    }
    if (len(p - o.c) < o.r) {
        return {};
    }
    double d = len(p - o.c);
    double r = o.r;
    double x = (d * d - r * r) / d;
    Point<T> base = p + Point<T>(o.c - p).tolen(x);
    Point<T> seg(o.c - p);
    Point<T> norm(-seg.y, seg.x);
    double h = sqrt(r * r - (d - x) * (d - x));
    norm = norm.tolen(h);
    return { base + norm, base + norm * (-1) };
}

template <typename T>
double arcLen(const Circle<T>& o, const Point<T>& p1, const Point<T>& p2) {
    if (o.r == 0) {
        cout << 0 << '\n';
        return 0;
    }
    double a = abs(angle(p1 - o.c, p2 - o.c));
    return a * o.r;
}

template <typename T>
struct Polygon {
    vector<Point<T>> p;
    Polygon() :p(vector<Point<T>>()) {}
    Polygon(const vector<Point<T>>& p) :p(p) {}
    size_t size() const {
        return p.size();
    }
    void resize(int n) {
        p.resize(n);
    }
    Point<T> operator [](int i) const {
        return p[i];
    }
    Point<T>& operator [](int i) {
        return p[i];
    }
    Point<T> back() const {
        return p.back();
    }
    Point<T>& back() {
        return p.back();
    }
    void push_back(const Point<T>& pt) {
        p.push_back(pt);
    }
    void pop_back() {
        p.pop_back();
    }
    bool isConvex() const {
        int pos = 0, neg = 0;
        for (size_t i = 0; i < size(); ++i) {
            int signCp = sign(cP(p[(i + 1) % size()] - p[i], p[i] - p[(i + size() - 1) % size()]));
            if (signCp == 1) {
                pos++;
            } else if (signCp == -1) {
                neg++;
            }
        }
        return sign(pos) * sign(neg) == 0;
    }
    double square() const {
        T sq = 0;
        for (size_t i = 0; i < size(); ++i) {
            sq += cP(p[(i + 1) % size()], p[i]);
        }
        return abs(sq) / 2.0;
    }
    double perimeter() const {
        double per = 0;
        for (size_t i = 0; i < size(); ++i) {
            per += len(p[(i + 1) % size()] - p[i]);
        }
        return per;
    }
    bool pointInPolygon(const Point<T>& pt) const {
        Point<T> far(1e9 - 123, 1e9 + 321);
        int cntCross = 0;
        for (size_t i = 0; i < size(); ++i) {
            if (pointOnSegment(pt, p[i], p[(i + 1) % size()])) {
                return true;
            }
            if (doSegmentsCross(pt, far, p[i], p[(i + 1) % size()])) {
                cntCross++;
            }
        }
        return cntCross % 2 == 1;
    }
    bool pointInConvexPolygon(const Point<T>& pt) {
        if (sign(cP(p[1] - p[0], p[size() - 1] - p[0])) == -1) {
            reverse(p.begin() + 1, p.end());
        }
        if (!pointInAngle(pt, p[1], p[0], p[size() - 1])) {
            return false;
        }
        int l = 1, r = size() - 1;
        while (l != r - 1) {
            int m = (l + r) / 2;
            if (sign(cP(p[m] - p[0], pt - p[0])) != -1) {
                l = m;
            } else {
                r = m;
            }
        }
        return pointInAngle(pt, p[0], p[l], p[l + 1]);
    }
};

template <typename T>
Polygon<T> convexHullJarvis(const vector<Point<T>>& p) {
    int n = p.size();
    if (n == 1) {
        return Polygon<T>(p);
    }
    Polygon<T> ch;
    int cur = min_element(p.begin(), p.end()) - p.begin();
    int start = cur;
    do {
        ch.push_back(p[cur]);
        int opt = 0;
        if (cur == 0) {
            opt++;
        }
        for (int i = 0; i < n; ++i) {
            if (p[i] != p[cur]) {
                int signCp = sign(cP(p[i] - p[cur], p[opt] - p[cur]));
                if (signCp == 1 || (signCp == 0 && len(p[i] - p[cur]) > len(p[opt] - p[cur]))) {
                    opt = i;
                }
            }
        }
        cur = opt;
    } while (p[cur] != p[start]);
    return ch;
}

template <typename T>
Polygon<T> convexHullGraham(vector<Point<T>>& p) {
    int n = p.size();
    int cur = min_element(p.begin(), p.end()) - p.begin();
    swap(p[0], p[cur]);
    sort(p.begin() + 1, p.end(), [&](const Point<T>& a, const Point<T>& b) -> bool {
        int signCp = sign(cP(a - p[0], b - p[0]));
        return signCp > 0 || (signCp == 0 && len(a - p[0]) < len(b - p[0]));
        });
    Polygon<T> ch;
    ch.push_back(p[0]);
    for (int i = 1; i < n; ++i) {
        if (p[i] == ch.back()) {
            continue;
        }
        while (ch.size() >= 2 && sign(cP(ch[ch.size() - 1] - ch[ch.size() - 2], p[i] - ch.back())) <= 0) {
            ch.pop_back();
        }
        ch.push_back(p[i]);
    }
    return ch;
}

template <typename T>
istream& operator >>(istream& in, Polygon<T>& p) {
    if (p.size() == 0) {
        int n;
        in >> n;
        p.resize(n);
    }
    for (size_t i = 0; i < p.size(); ++i) {
        in >> p[i];
    }
    return in;
}

struct Triplet {
    Point<ll> p1, p2, p3;
    bool operator<(const Triplet& o) const {
        return make_pair(p1, make_pair(p2, p3)) < make_pair(o.p1, make_pair(o.p2, o.p3));
    }
    bool operator==(const Triplet& o) const {
        return p1 == o.p1 && p2 == o.p2 && p3 == o.p3;
    }
    bool operator!=(const Triplet& o) const {
        return !(*this == o);
    }
    Triplet(Point<ll> p1, Point<ll> p2, Point<ll> p3) {
        Point<ll> a[3] = { p1, p2, p3 };
        sort(a, a + 3);
        this->p1 = a[0];
        this->p2 = a[1];
        this->p3 = a[2];
    }
};


template<typename T>
double dist(int l, int r, const vector<Point<T>> &p){
    if (r - l < 2 || l >= r)
        return 2e8 + 10;
    int m = (l + r) / 2;
    if (l + 1 == m && m + 1 == r)
        return (p[m] - p[l]).len();
    double d = min(dist(l, m, p), dist(m, r, p));
    auto it1 = upper_bound(p.begin() + l, p.begin() + r, int(floor(p[m].x - d)), [](int x, const Point<T> &p) -> bool{
        return p.x <= x;
    });
    auto it2 = upper_bound(it1, p.begin() + r, int(ceil(p[m].x + d)), [](int x, const Point<T> &p) -> bool{
        return p.x < x;
    });
    if (it1 == p.begin() + r)
        return d;
    vector<Point<T>> b(it1, it2);
    sort(all(b), [](const Point<T> &l, const Point<T> &r){
        return l.y < r.y;
    });
    for (int i = 0; i < b.size(); ++i){
        int j = i - 1;
        while (j >= 0 && b[i].y - b[j].y < d)
            d = min(d, (b[i] - b[j]).len());
        j = i + 1;
        while (j < b.size() && b[j].y - b[i].y < d)
            d = min(d, (b[i] - b[j]).len());
    }
    return d;
}

int main(){
    ios_base::sync_with_stdio(false);
    cin.tie(0);
    int n;
    cin >> n;
    vector<Point<ll>> a(n);
    for (auto &x : a)
        cin >> x;
    ll maxd = 0;
    pair<int, int> longest = {0, 0};
    for (int i = 0; i < n; ++i)
        for (int j = i + 1; j < n; ++j) {
            auto d = abs(a[i].len2() - a[j].len2());
            if (d > maxd){
                longest = {i, j};
                maxd = d;
            }
        }
    auto p1 = a[longest.first], p2 = a[longest.second];
    //cout << abs(p1.len2() - p2.len2()) << '\n';
    Line l1(p1, p2);
    auto norm = l1.norm();
    Line l2(p1, p1 + norm);
    Line l3(p2, p2 + norm);
    cout << l2.A << ' ' << l2.B << ' ' << l2.C << '\n';
    cout << l3.A << ' ' << l3.B << ' ' << l3.C << '\n';
}