#include <cstdio>
#include <cstdint>
#include <cctype>
#include <cmath>
#include <vector>
#include <string>
#include <functional>

template<typename T>
constexpr T seqSum(T first, T last, T step = 1) noexcept {
	return ((first + last) * ((last - first + step) / step)) / 2;
}

template<size_t N>
constexpr size_t rowsSize(size_t row) noexcept {
	if (row == 0)
		return 0;
	row -= 1;
	return row < N
		? seqSum(N, N + row)
		: seqSum(N, 2 * N - 1) + seqSum(2 * N - row + 1, 2 * N - 2);
}

template<size_t N>
constexpr size_t oneRowLen(size_t row) noexcept {
	return row < N
		? N + row
		: 2 * N - row + 1;
}

static_assert(rowsSize<3>(5) == 19);

template<size_t N>
struct Point final {
	size_t row = 0, col = 0;
};

struct CubePoint final {
	long long x = 0, y = 0, z = 0;

	template<size_t N>
	static CubePoint make(size_t row, size_t col) noexcept {
		CubePoint r { };
		long long right = (long long)(col)-(N - 1);
		r.y -= right;
		r.x += right;
		if (row >= N) {
			long long down = (long long)(row)-(N - 1);
			r.y -= down;
			r.z += down;
		} else {
			long long up = (N - 1) - row;
			r.x += up;
			r.z -= up;
		}
		return r;
	}

	template<size_t N>
	void toOffset(size_t& row, size_t& col) noexcept {
		row = size_t(-(y + x) + (N - 1));
		if (row >= N) {
			col = size_t(-(y + z) + (N - 1));
		} else {
			col = size_t((x + z) + (N - 1));
		}
	}
};

struct Op final {
	uint8_t rot = 0; // mod 6
	bool inv = false;

	void operator+=(uint8_t s) & {
		rot = (rot + s) % 6;
	}

	void operator!() & {
		inv = !inv;
		rot = (6 - rot) % 6;
	}

	template<size_t N>
	Point<N> operator*(Point<N> p) const {
		auto cube = CubePoint::make<N>(p.row, p.col);
		if (inv)
			std::swap(cube.x, cube.y);
		for (int i = 0; i < rot; ++i) {
			long long tmp = cube.x;
			cube.x = -cube.y;
			cube.y = -cube.z;
			cube.z = -tmp;
		}
		cube.template toOffset<N>(p.row, p.col);
		return p;
	}
};

template<size_t N>
struct Hex final {
	constexpr static size_t SIZE = rowsSize<N>(2 * N - 1);
	std::vector<uint8_t> data;

	Hex() : data(SIZE, 0) { }

	uint8_t& operator()(size_t row, size_t col) noexcept { // Global coords
		return data[rowsSize<N>(row) + col];
	}

	static Hex read() {
		Hex result;
		for (size_t row = 0; row < 2 * N - 1; ++row) {
			size_t rowSize = oneRowLen<N>(row);
			for (size_t col = 0; col < rowSize; ++col) {
				unsigned tmp;
				scanf("%u", &tmp);
				result(row, col) = static_cast<uint8_t>(tmp);
			}
		}
		return result;
	}

	void print() {
		for (size_t row = 0; row < 2 * N - 1; ++row) {
			size_t rowSize = oneRowLen<N>(row);
			for (size_t col = 0; col < rowSize; ++col)
				printf("%u ", unsigned{ this->operator()(row, col) });
			putchar('\n');
		}
		fflush(stdout);
	}
};

Op readOps() {
	char ch;
	while (isspace(ch = getchar()));
	Op op;
	do {
		switch (ch) {
			case 'L': op += 1; break;
			case 'R': op += 5; break;
			case 'T': !op; break;
		default: break;
		}
	} while ((ch = getchar()) == 'L' || ch == 'T' || ch == 'R');
	return op;
}

template<size_t N>
void solve(Hex<N>& hex, Op op) {
	Hex<N> result;
	for (size_t row = 0; row < 2 * N - 1; ++row) {
		size_t rowSize = oneRowLen<N>(row);
		for (size_t col = 0; col < rowSize; ++col) {
			Point<N> p = op * Point<N> { row, col };
			result(p.row, p.col) = hex(row, col);
		}
	}

	result.print();
}

void dispatch(size_t n) {
#define D(n) case n:{ Hex<n> data = Hex<n>::read(); solve<n>(data, readOps()); } break;
#define D1(n) D(n) D(n + 1) D(n + 2) D(n + 3)
#define D2(n) D1(n) D1(n + 4) D1(n + 8) D1(n + 12)
#define D3(n) D2(n) D2(n + 16) D2(n + 32) D2(n + 48)
#define D4(n) D3(n) D3(n + 64) D3(n + 128) D3(n + 192)
	switch (n) {
		D4(2);
		D4(258);
	default: break;
	}
#undef D4
#undef D3
#undef D2
#undef D1
#undef D
}

int main() {
	long long unsigned n;
	scanf("%llu", &n);
	dispatch(static_cast<size_t>(n));
	return 0;
}